<?php

defined('ABSPATH') || exit();

class WC_Payrex_Controller_Webhook
{
    /**
     * Method to register the routes to WooCommerce
     */
    public function register_routes()
    {
        register_rest_route('wc-payrex/v1', 'webhooks', [
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => [ $this, 'process_webhook' ],
            'permission_callback' => '__return_true',
            'args'                => [
                'mode'          => [
                    'required' => true,
                    'validate_callback' => function ($param) {
                        if ($param !== 'mode' && $param !== 'test') {
                            return new WP_Error(
                                'parameter_invalid',
                                'Parameter value invalid.',
                                [ 'status' => 400 ]
                            );
                        }

                        return true;
                    }
                ]
            ],
        ]);
    }

    public function process_webhook($request)
    {
        $payload = $request->get_body();
        $payload = @file_get_contents('php://input');
        $signatureHeader = isset($_SERVER['HTTP_PAYREX_SIGNATURE']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_PAYREX_SIGNATURE'])) : null;

        $mode = $request['mode'];

        $settings = get_option('woocommerce_payrex_settings', []);

        $webhook_secret = $mode == 'live' ? $settings['live_webhook_secret'] : $settings['test_webhook_secret'];

        try {
            $event = \Payrex\Webhook::parseEvent(
                $payload,
                $signatureHeader,
                $webhook_secret
            );
        } catch (\Payrex\Exceptions\UnexpectedValueException $e) {
            return new WP_Error('payload_invalid', 'Error parsing payload.', [ 'status' => 400 ]);
        } catch (\Payrex\Exceptions\SignatureVerificationException $e) {
            return new WP_Error('payload_invalid', 'Error verifiying webhook signature', [ 'status' => 400 ]);
        } catch (Exception $e) {
            return new WP_Error('payload_invalid', 'Invalid webhook signature', [ 'status' => 400 ]);
        }

        if ($event->type === 'payment_intent.succeeded') {
            $intent = $event->data->resource;

            $query = new WC_Order_Query([
                'limit'        => 1,
                'return'       => 'objects',
                'status'       => 'any',
                'meta_key'     => '_payrex_intent_id',
                'meta_value'   => $intent->id,
            ]);

            $orders = $query->get_orders();

            if (empty($orders)) {
                return new WP_Error('resource_not_found', 'Order not found.', [ 'status' => 404 ]);
            }

            $order = $orders[0];
            $message = 'No action taken';

            if (!$order->is_paid()) {
                $order->payment_complete($intent->latest_payment['id']);
                $order->add_order_note('Payment succeeded via PayRex webhook. Intent ID: ' . $intent->id);
                $message = 'Woocommerce marked order as payment completed.';
            }

            return new WP_REST_Response([
                'status' => 'success',
                'message' => $message
            ]);
        } elseif ($event->type === 'refund.updated') {
            $refund = $event->data->resource;

            $query = new WC_Order_Query([
                'limit'        => 1,
                'return'       => 'objects',
                'status'       => 'any',
                'meta_key'     => '_payrex_refund_id',
                'meta_value'   => $refund->id,
            ]);

            $refunds = $query->get_orders();

            if (empty($refunds)) {
                return new WP_Error('resource_not_found', 'Refund not found.', [ 'status' => 404 ]);
            }

            $refund = $refunds[0];
            $order_id = $refund->get_parent_id();
            $order = wc_get_order($order_id);
            $message = "No action taken.";

            if ($refund->get_meta('_refund_pending') === 'yes') {
                if ($refund->status === 'failed') {
                    wp_delete_post($refund->get_id(), true);
                    $order->add_order_note("Refund failed. PayRex encountered an error.");
                    $message = "Woocommerce refund record deleted.";
                } elseif ($refund->status === 'succeeded') {
                    $refund->update_meta_data('_refund_pending', 'no');
                    $refund->save();
                    $order->add_order_note("Refunded {$refund->get_amount()} via Webhook. Refund ID: {$refund->id}.");
                    $message = "Woocommerce refund updated.";
                }
            }

            return new WP_REST_Response([
                'status' => 'success',
                'message' => $message
            ]);
        }

        return new WP_REST_Response(['status' => 'ignored']);
    }
}
