<?php

defined( 'ABSPATH' ) || exit;

class WC_Payrex_Extension_Loader
{
    /**
     * Loads the dependency files of the extension.
     */
    public function loadFiles()
    {
        require_once WC_GATEWAY_PAYREX_DIR_PATH . 'includes/class-wc-gateway-payrex.php';
        require_once WC_GATEWAY_PAYREX_DIR_PATH . 'includes/admin-settings.php';
        require_once WC_GATEWAY_PAYREX_DIR_PATH . 'includes/controllers/class-wc-payrex-controller-payment-intent.php';
        require_once WC_GATEWAY_PAYREX_DIR_PATH . 'includes/controllers/class-wc-payrex-controller-webhook.php';
        require_once WC_GATEWAY_PAYREX_DIR_PATH . 'includes/controllers/class-wc-payrex-controller-payment-callback.php';
    }

    /**
     * Initialize the payment gateway plugin
     */
    public function init()
    {
        add_filter('plugin_action_links_' . plugin_basename(WC_GATEWAY_PAYREX_FILE), 'add_payrex_plugin_action_links');
        add_action('admin_post_payrex_connect', [$this, 'connect_payrex_account_action']);
        add_action('admin_post_payrex_disconnect', [$this, 'disconnect_payrex_account_action']);
        // TODO: To check if this is still needed later.
        add_action('admin_post_payrex_setup_webhook', [$this, 'setup_payrex_webhook_action']);
        // Register payrex plugin to the list of payment gateways in woocommerce. required filter.
        add_filter('woocommerce_payment_gateways', [$this, 'add_payrex_gateway_action']);

        add_action('admin_notices', function () {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only flag for admin notice
            if (isset($_GET['error'])) {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Read-only admin notice flag
                $error = sanitize_text_field(wp_unslash($_GET['error']));

                if ($error === 'webhook_setup_failed') {
                    echo '<div class="notice notice-error"><p>Webhook setup failed, please make sure that your account is activated.</p></div>';
                }
            }
        });

        // Prepares the relevant routes for APIs. required actions
        add_action('rest_api_init', function () {
            if (class_exists('WooCommerce')) {
                WC()->session = new WC_Session_Handler();
                WC()->session->init();
            }

            $paymentIntentController = new WC_Payrex_Controller_Payment_Intent();
            $webhookController = new WC_Payrex_Controller_Webhook();
            $paymentCallback = new WC_Payrex_Controller_Payment_Callback();

            $paymentIntentController->register_routes();
            $webhookController->register_routes();
            $paymentCallback->register_routes();
        });

        add_action('woocommerce_blocks_loaded', [$this, 'register_payment_method_type_action']);

        add_action('wp_enqueue_scripts', function () {
            if (is_checkout()) {
                wp_enqueue_style(
                    'woocommerce_gateway_payrex_style',
                    plugin_dir_url(WC_GATEWAY_PAYREX_FILE) . 'build/style.css',
                    [],
                    filemtime(WC_GATEWAY_PAYREX_DIR_PATH . 'build/style.css')
                );
            }
        });

        add_action('before_woocommerce_init', [$this, 'set_compatibility_action']);
    }

    public function set_compatibility_action()
    {
        if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
                'cart_checkout_blocks',
                WC_GATEWAY_PAYREX_FILE,
                true
            );

            \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
                'custom_order_tables',
                WC_GATEWAY_PAYREX_FILE,
                true
            );
        }
    }

    /**
     * Method used to register the payrex payment gateway in WooCommerce
     */
    public function add_payrex_gateway_action($methods)
    {
        $methods[] = 'WC_Gateway_PayRex';

        return $methods;
    }

    /**
     * Register the payment block
     */
    public function register_payment_method_type_action()
    {
        if (!class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
            return;
        }

        require_once WC_GATEWAY_PAYREX_DIR_PATH . 'includes/Blocks/Payments/PayrexBlock.php';

        add_action(
            'woocommerce_blocks_payment_method_type_registration',
            function (Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry) {
                if (get_woocommerce_currency() === 'PHP') {
                    $payment_method_registry->register(new PayrexBlock());
                }
            }
        );
    }

    /**
     * Triggers when the wordpress user disconnects their PayRex account
     */
    public function disconnect_payrex_account_action()
    {
        if (!current_user_can('manage_woocommerce')) {
            wp_die('Permission denied');
        }

        $adminReferrerChecker = check_admin_referer('payrex_disconnect');
        if($adminReferrerChecker === false) {
            wp_die('Invalid referrer');
        }

        $settings = $this->getExtensionSettings();

        $this->deletePayrexWebhook('live', $settings);
        $this->deletePayrexWebhook('test', $settings);

        update_option('woocommerce_payrex_settings', []);

        wp_safe_redirect(admin_url('admin.php?page=wc-settings&tab=checkout&section=payrex'));
        exit;
    }

    /**
     * Triggers when the user has successfully validated to connect their PayRex merchant.
     */
    public function connect_payrex_account_action()
    {
        if (! current_user_can('manage_woocommerce')) {
            wp_die('Permission denied');
        }

        $adminReferrerChecker = check_admin_referer('payrex_connect');
        if($adminReferrerChecker === false) {
            wp_die('Invalid referrer');
        }

        $token = isset($_GET['token']) ? sanitize_text_field(wp_unslash($_GET['token'])) : null;

        if (!$token) {
            return new WP_REST_Response(
                [
                    'error' => 'Missing token'
                ],
                400
            );
        }

        $parts = explode('.', $token);

        if (count($parts) !== 3) {
            return new WP_Error('invalid_jwt', 'Invalid token.');
        }

        $payload = $parts[1];

        $credentials = json_decode($this->base64urlDecode($payload), true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('invalid_json', 'Invalid token.');
        }

        $settings = $this->getExtensionSettings();

        $settings['live_secret_key'] =  $credentials['live']['secret_key'];
        $settings['live_publishable_key'] =  $credentials['live']['publishable_key'];
        $settings['test_secret_key'] =  $credentials['test']['secret_key'];
        $settings['test_publishable_key'] =  $credentials['test']['publishable_key'];
        $settings['merchant_id'] =  $credentials['live']['merchant_id'];
        $settings['trade_name'] =  $credentials['live']['trade_name'];

        $live_webhook = $this->createPayrexWebhook($settings['live_secret_key'], 'live');
        $test_webhook = $this->createPayrexWebhook($settings['test_secret_key'], 'test');

        $settings['live_webhook_id'] =  $live_webhook['id'];
        $settings['live_webhook_secret'] = $live_webhook['secret_key'];
        $settings['test_webhook_id'] =  $test_webhook['id'];
        $settings['test_webhook_secret'] = $test_webhook['secret_key'];

        $this->updateExtensionSettings($settings);

        wp_safe_redirect(
            admin_url('admin.php?page=wc-settings&tab=checkout&section=payrex')
        );
        exit;
    }

    // TODO: To check if this is still needed later.
    public function setup_payrex_webhook_action()
    {
        if (! current_user_can('manage_woocommerce')) {
            wp_die('Permission denied');
        }

        $adminReferrerChecker = check_admin_referer('payrex_setup_webhook');
        if($adminReferrerChecker === false) {
            wp_die('Invalid referrer');
        }

        $mode = isset($_GET['mode']) ? sanitize_text_field(wp_unslash($_GET['mode'])) : null;

        if (!$mode) {
            return new WP_REST_Response(
                [
                    'error' => 'Missing parameter'
                ],
                400
            );
        }

        if ($mode != 'live' && $mode != 'test') {
            return new WP_Error('invalid_parameter', 'Invalid mode.');
        }

        $settings = $this->getExtensionSettings();
        $secret_key = $mode === 'live' ? $settings['live_secret_key'] : $settings['test_secret_key'];

        $webhook = $this->createPayrexWebhook($secret_key, $mode);

        if (empty($webhook)) {
            wp_safe_redirect(admin_url('admin.php?page=wc-settings&tab=checkout&section=payrex&error=webhook_setup_failed'));
            exit;
        }

        if ($mode === 'live') {
            $settings['live_webhook_id'] =  $webhook['id'];
            $settings['live_webhook_secret'] = $webhook['secret_key'];
        } else {
            $settings['test_webhook_id'] =  $webhook['id'];
            $settings['test_webhook_secret'] = $webhook['secret_key'];
        }

        $this->updateExtensionSettings($settings);

        wp_safe_redirect(
            admin_url('admin.php?page=wc-settings&tab=checkout&section=payrex')
        );
        exit;
    }

    private function createPayRexWebhook($secret_key, $mode)
    {
        $webhook_url = home_url('/wp-json/wc-payrex/v1/webhooks?mode=' . $mode);

        $client = new \Payrex\PayrexClient($secret_key);

        try {
            $webhooks = $client->webhooks->list(
                [
                    'url' => $webhook_url,
                ]
            );
        } catch (\Payrex\Exceptions\AuthenticationException $e) {
            return null;
        }

        if (count($webhooks->data) > 0) {
            $webhook = $webhooks->data[0];

            return [
                'id' => $webhook->id,
                'secret_key' => $webhook->secret_key
            ];
        }

        try {
            $baseUrl = home_url();

            $webhook = $client->webhooks->create([
                'url' => $webhook_url,
                'description' => "WooCommerce Webhook for $baseUrl",
                'events' => [
                    'payment_intent.succeeded',
                    'refund.updated',
                ],
            ]);
        } catch (Exception $e) {
            return null;
        }

        return [
            'id' => $webhook->id,
            'secret_key' => $webhook->secret_key
        ];
    }

    /**
     * Delete the PayRex webhook
     */
    private function deletePayrexWebhook($mode, $settings)
    {
        $webhookId = $settings["{$mode}_webhook_id"];

        if (empty($webhookId)) {
            return;
        }

        $client = new \Payrex\PayrexClient($settings["{$mode}_secret_key"]);

        try {
            $client->webhooks->delete($webhookId);
        } catch (\Payrex\Exceptions\ResourceNotFoundException $e) {
            return null;
        }
    }

    private function base64urlDecode($data)
    {
        $data = strtr($data, '-_', '+/');
        $padding = strlen($data) % 4;
        if ($padding > 0) {
            $data .= str_repeat('=', 4 - $padding);
        }

        return base64_decode($data);
    }

    /**
     * Returns the current installed extension settings
     */
    private function getExtensionSettings()
    {
        return get_option('woocommerce_payrex_settings', []);
    }

    /**
     * Sets the current installed extension settings
     */
    public function updateExtensionSettings($settings)
    {
        update_option('woocommerce_payrex_settings', $settings);
    }
}
