<?php

use Automattic\WooCommerce\Blocks\Assets\Api;
use Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType;

require_once WC_GATEWAY_PAYREX_DIR_PATH . 'includes/class-wc-token.php';

/**
 * This is the server-side class used to handle server-side logic of the PayRex integration to the merchant's website.
 *
 */
final class PayrexBlock extends AbstractPaymentMethodType
{
    private $gateway;

    /**
     * Payment method name that matches the id from the client-side integration.
     *
     * @var string
     */
    protected $name = "payrex";

    public function initialize()
    {
        $this->settings = get_option('woocommerce_payrex_settings', []);

        $gateways = WC()->payment_gateways->payment_gateways();
        $this->gateway = $gateways[$this->name];
    }

    /**
     * Extended from the parent class.
     */
    public function is_active()
    {
        return filter_var($this->get_setting('enabled', false), FILTER_VALIDATE_BOOLEAN);
    }

    /**
     * Returns an array of scripts/handles to be registered for this payment method. Extended from the parent class.
     *
     * @return array
     */
    public function get_payment_method_script_handles()
    {
        wp_register_script(
            'payrex',
            'https://js.payrexhq.com',
            [],
            time(),
            true
        );

        wp_register_script(
            'payrex-instance',
            plugin_dir_url(WC_GATEWAY_PAYREX_FILE) . 'assets/payrex.js',
            ['payrex'],
            time(),
            true
        );

        $publishableKey = $this->gateway->getPublishableKey();

        wp_localize_script('payrex-instance', 'gateway_data', ['publishableKey' => $publishableKey]);

        wp_register_script(
            'wc-payment-method-payrex',
            plugin_dir_url(WC_GATEWAY_PAYREX_FILE) . 'build/index.js',
            [
                'wc-blocks-registry',
                'wc-settings',
                'wp-element',
                'wp-html-entities',
                'wp-i18n',
                'payrex',
                'payrex-instance'
            ],
            time(),
            true
        );

        wp_localize_script('wc-payment-method-payrex', 'payrex_payment_method_data', [
            'token' => Wc_Token::generate_create_payment_intent_token(),
            'isTestMode' => !$this->gateway->isLivemode()
        ]);

        return ['payrex', 'payrex-instance', 'wc-payment-method-payrex'];
    }

    /**
     * Returns an array of key=>value pairs of data made available to the payment methods script. Extended from the parent class.
     *
     * @return array
     */
    public function get_payment_method_data()
    {
        return [
            'title'                    => $this->gateway->title,
            'description'              => $this->gateway->description,
            'payment_method_icons'     => $this->gateway->getEnabledPaymentMethodIcons(),
            'icon' => $this->gateway->icon
        ];
    }
}
